function circulation_features_archive

% 1) LOAD DATA
[eke_lats, eke_vals] = get_eke_data;
[D_lats  , D_vals  ] = get_D_data;
[psi_lats, psi_vals] = get_psi_data;
    %EKE: 9 exps x AQ,JJA x zm   (MJ/m^2)
    %D:   9 exps x AQ,JJA        (Tg/s)
    %psi: 9 exps x AQ,JJA x zivr (Tg/s)

% 2) COMPUTE CORRELATIONS AND FITS
eke_all = cat(3,eke_lats,eke_vals);%9 exps x AQ,JJA x lats,vals
D_all   = cat(3,D_lats  ,D_vals  );
psi_all = cat(3,psi_lats,psi_vals);
p_eke_all = NaN(2,2,2); %2 coeffs (p') x AQ,JJA x lats,vals
p_psi_all = NaN(2,2,2); %2 coeffs (p') x AQ,JJA x lats,vals
r_eke_all = NaN(1,2,2); %1             x AQ,JJA x lats,vals
r_psi_all = NaN(1,2,2); %1             x AQ,JJA x lats,vals
l_eke_all = NaN(2,2,2); %[0 100]       x AQ,JJA x lats,vals
l_psi_all = NaN(2,2,2); %[0 100]       x AQ,JJA x lats,vals
for kk = 1:2 %lats,vals
for ii = 1:2 %AQ,JJA
    [p_eke_all(:,ii,kk), r_eke_all(:,ii,kk)] = compute_cf(D_all(:,ii,kk),eke_all(:,ii,kk));
    [p_psi_all(:,ii,kk), r_psi_all(:,ii,kk)] = compute_cf(D_all(:,ii,kk),psi_all(:,ii,kk));
    l_eke_all(:,ii,kk) = polyval(p_eke_all(:,ii,kk),[0; 100]);
    l_psi_all(:,ii,kk) = polyval(p_psi_all(:,ii,kk),[0; 100]);
        %polyval cares how x (but not p) is oriented--x range of [0 100]
        %covers all relevant diffusivity lats and intensities
end
end
y_all = cat(4,eke_all  ,psi_all);  %9 exps        x AQ,JJA x lats,vals x eke,psi
p_all = cat(4,p_eke_all,p_psi_all);%2 coeffs (p') x AQ,JJA x lats,vals x eke,psi
r_all = cat(4,r_eke_all,r_psi_all);%1             x AQ,JJA x lats,vals x eke,psi
l_all = cat(4,l_eke_all,l_psi_all);%[0 100]       x AQ,JJA x lats,vals x eke,psi

% 3) SET FIGURE PARAMETERS
%    Horizontal half page, 19.0 cm wide x 11.5 cm tall
fwidth   = 19; %Figure width
fheight  = 11.5; %Figure height
bmargin  = .9; %Bottom margin
hmargin  = .5; %Horizontal margin (between rows)
tmargin  = .25; %Top margin
lmargin  = 1.1; %Left margin
vmargin  = 1.2; %Vertical margin (between columns)
rmargin  = .25;%Right margin
pheight  = 0.5*(fheight-bmargin-hmargin-tmargin); %Panel height
pwidth   = 0.5*(fwidth-lmargin-vmargin-rmargin);%Panel width

% 4) SET AXES PARAMETERS
fontsize = 8; %Requested by AGU
xlims  = {[32 48],[10 18]};
xticks = {32:2:48, 10:18};
ylims  = {[31 52], [.4 1.1];[29 43],[58 79]};
yticks = {31:3:52,.4:.1:1.1;29:2:43,58:3:79};
panel_labs = {'(a)','(b)';'(c)','(d)'}; %One per panel
panel_xval = [33.5 10.75];
panel_yval = [49 1; 41 76];
ctrlnames = {'AQUA','AMIP'};
fl_styles = {'-','--'};
mshapes   = {'o','s'};
xldesc    = {'Latitude of D maximum (deg)','Intensity of D maximum (Tg/s)'};
yldesc    = {'Latitude of EKE maximum (deg)'                ,'Intensity of EKE maximum (MJ/m^2)';...
             'Latitude of p_s\psi_{RMS}cos\phi/g max (deg)','Intensity of p_s\psi_{RMS}cos\phi/g max (Tg/s)'};
%l_pos = {[3.5 10 4 1]   ,[13.5 8.5 4 1];...
%         [3.5 4.5 4 .67],[12.5 4.5 4 .67]};
cmap = parula(9);
xfp  = cat(3,[35 11.5;   34 10.2],[35 11.5; 34     10.2]); %row x col x AQ,JJA
yfp  = cat(3,[34  1.067; 31 61  ],[32  1  ; 29.667 59  ]);
grayvec = [.55 .55 .55];
               
% 5) CHECK IF FIGURE EXISTS
%sname = mfilename;
%sname = sname(22:end); %Get version number
%plot_dir = '/project/tas1/tmooring/plots/echam_diffusivity_plots/';
%if exist([plot_dir 'circulation_features_'  sname '.ps'],'file')
%    error('Planned output plot already exists!')
%end

% 6) DRAW FIGURE
axpos_all = {[lmargin     bmargin+pheight+hmargin pwidth pheight],...
  [lmargin+pwidth+vmargin bmargin+pheight+hmargin pwidth pheight];...
             [lmargin     bmargin                 pwidth pheight],...
  [lmargin+pwidth+vmargin bmargin                 pwidth pheight]};
axarr = NaN(2,2);
fh = figure('units','centimeters','position',[3 3 fwidth fheight],...
    'paperunits','centimeters','papersize',[fwidth fheight],...
    'paperposition',[0 0 fwidth fheight],'color','w');
for ii = 1:2 %rows (EKE, psi_RMS)
for jj = 1:2 %cols (lat, intensity)
    
% a) Create panel and draw lines for use in legend generation
    axarr(ii,jj) = axes('units','centimeters','position',axpos_all{ii,jj});
if (ii==2) && (jj==2) %lower right
for ll = 1:9 %9 exps
    plot(0,0,'linestyle','none','marker','o','color',cmap(ll,:),...
        'markerfacecolor',cmap(ll,:));
    hold on
end %9 exps
    plot(0,0,'o-' ,'color',grayvec,'markerfacecolor',grayvec)
    plot(0,0,'s--','color',grayvec,'markerfacecolor',grayvec)
end
    
% b) Put actual data on panel
for kk = 1:2 %AQ,JJA
    plot([0 100]',l_all(:,kk,jj,ii),'color',grayvec,'linestyle',fl_styles{kk})
    hold on
for ll = 1:9 %9 exps
    plot(D_all(ll,kk,jj),y_all(ll,kk,jj,ii),'linestyle','none','marker',...
        mshapes{kk},'color',cmap(ll,:),'markerfacecolor',cmap(ll,:));
end %9 exps
end %AQ,JJA

% c) Resize panel and add appropriate labels
    set(axarr(ii,jj),'position',axpos_all{ii,jj},'xlim',xlims{jj},...
        'xtick',xticks{jj},'fontsize',fontsize,'ylim',ylims{ii,jj},...
        'ytick',yticks{ii,jj})
if ii==2 %bottom row    
    xlabel(xldesc{jj},'fontsize',fontsize)
else
    set(axarr(ii,jj),'xticklabel',[])
end
if (ii==2) && (jj==2) %lower right
    lh = legend('-8K','-6K','-4K','-2K','Control','+2K','+4K','+6K','+8K',...
        'AQUA','AMIP');
    legend boxoff
    set(lh,'units','centimeters','position',[16.3 1.4 2.4 3.75]);
end
for kk = 1:2 %AQ,JJA
    if p_all(2,kk,jj,ii)>0
        csgn = '+';
        cstr = num2str(p_all(2,kk,jj,ii),4);
    elseif p_all(2,kk,jj,ii)<0
        csgn = '-';
        cstr = num2str(p_all(2,kk,jj,ii),4); %Do it this way so we properly
        cstr = cstr(2:end); %round the signed constant, rather than taking
            %the magnitude of the constant before rounding
    else
        error('Constant is unexpectedly zero!')
    end
    text(xfp(ii,jj,kk),yfp(ii,jj,kk),[ctrlnames{kk} ...
        ': r = ' num2str(r_all(:,kk,jj,ii),4) ...
        ', y = ' num2str(p_all(1,kk,jj,ii),4) 'x ' csgn ' ' cstr],...
        'fontsize',fontsize)
end %AQ,JJA
    ylabel(yldesc{ii,jj},'fontsize',fontsize)
    text(panel_xval(jj),panel_yval(ii,jj),panel_labs{ii,jj},'fontsize',...
        fontsize)
end %cols (lat, intensity)
end %rows (EKE, psi_RMS)

% d) Save figure
pause(1)
set(fh,'position',[3 3 fwidth fheight])
pause(1)
for ii = 1:2 %rows
for jj = 1:2 %cols
    set(axarr(ii,jj),'position',axpos_all{ii,jj})
    pause(1)
end
end
set(gcf,'position',[3 3 fwidth fheight])
pause(1)
print(fh,'-dpsc','circulation_features_archive.ps');
pause(1)
close(fh)

% D) COMPUTE LINEAR FITS AND CORRELATIONS
function [p, r] = compute_cf(D,y)

% Use diffusivity quantity as predictor, compute best-fit line and corr

p = polyfit(D,y,1)';
r = corr(D,y);
        
% C) GET PSI_RMS DATA
function [psir_max_lat, psir_max_val] = get_psi_data

wsdir  = '';
load([wsdir 'ECHAM_AMIP_925hPa_psi_RMS_zm_zi_may16_2019.mat']);
psir_max_lat = psir_max_lat(:,:,2);
psir_max_val = psir_max_val(:,:,2);

% B) GET D DATA
function [Dmx_lat, Dmx_val] = get_D_data

% After circulation_features_v2

wsdir  = '';
aqua_D = load([wsdir 'tiffany_ECHAM16K_Dmax_attrib_jul10_2018.mat']);
amip_D = load([wsdir 'tiffany_AMIP16K_Dmax_attrib_jul09_2018.mat']);
    %For diffusivity, latitudes increase w/ increasing index

Dmx_lat = NaN(2,9); %Lats of D maxima, (AQUA, AMIP) x 9 exps
Dmx_val = NaN(2,9); %Strengths of D maxima
Dmx_lat(1,:) =  aqua_D.max_lats_vals(1,:,1); %deg S
Dmx_lat(2,:) = -amip_D.max_lats_vals(1,:,1);
Dmx_val(1,:) =  aqua_D.max_lats_vals(1,:,2); %kg/s
Dmx_val(2,:) =  amip_D.max_lats_vals(1,:,2);
Dmx_lat      =  Dmx_lat'; %9 exps x AQ,JJA
Dmx_val      =  Dmx_val'/1e9; %Tg/s
    
% A) GET EKE DATA
function [eke_max_lat, eke_max_val] = get_eke_data

% After circulation_features_v2

% 1) LOAD DATA
wsdir = '';
aqua_ke_ws = load([wsdir 'ECHAM20K_col_int_eke_comps_jun27_2018.mat']);
amip_ke_ws = load([wsdir 'AMIP16K_col_int_eke_comps_jun29_2018.mat']);
    %For EKE, latitudes decrease w/ increasing index--lats are same for
    %both workspaces

eke_aqua_raw = squeeze(0.5*sum(mean(mean(aqua_ke_ws.xp2(:,:,:,:,2:10),...
    2),3),4));
    %96 lats x 12 months x 10 yrs x 2 winds x 11 exps to start, average and
    %squeeze to 96 lats x 9 exps.  Will symmetrize after regridding to 0.1
    %deg, in keeping with tiffany_ECHAM16K_Dmax_attrib_jul10_2018
eke_amip_raw = squeeze(0.5*sum(mean(amip_ke_ws.xp2,3),4)); %96 lats x 12
    %mons x 30 yrs x 2 winds x 9 exps to start, average and squeeze to 96
    %lats x 12 mons x 9 exps
days_jja = [30 31 31];
eke_amip_raw = squeeze(sum(bsxfun(@times,days_jja,eke_amip_raw(:,6:8,...
    :)),2))./sum(days_jja); %Subset to JJA, take suitably weighted mean,
    %and squeeze to 96 lats x 9 exps
    
lat_raw = aqua_ke_ws.lat; %Load lats--plots of both EKEs on native grids
    %show generally larger EKEs in AQUA than in AMIP and much stronger EKE
    %in SH than in NH for AMIP.  EKE generally increases with SST
    
% 2) PROCESS DATA
eke_raw = cat(3,eke_aqua_raw,eke_amip_raw);%96 lats x 9 exps x (AQUA, AMIP)
lat_hi  = (90:-.1:-90)';
eke_0p1 = interp1(lat_raw,eke_raw,lat_hi,'spline'); %1801 lats x 9 exps x
    %(AQUA, AMIP)
eke_0p1(:,:,1) = 0.5*(eke_0p1(:,:,1)+flip(eke_0p1(:,:,1),1)); %Symmetrize
eke_0p1 = flip(eke_0p1,1); %Lat now increases w/ increasing index
lat_hi  = flip(lat_hi,1);
%eke_0p1 = repmat(eke_0p1,[1 1 1 2]); %1801 lats x 9 exps x (AQUA, AMIP) x
    %(zonal mean, zonal integral)
%eke_0p1(:,:,:,2) = bsxfun(@times,2*pi*todd_constants('a')*cosd(lat_hi),...
%    eke_0p1(:,:,:,2)); %Take zonal integral, switching units from J/m^2 to
    %J/m--this zonally integrated EKE is arguably more conceptually
    %analogous to the diffusivity than the native zonal mean EKE.  Plots of
    %zonal mean EKE have all the expected features and overall amplitude
    %seems reasonable.  Should be acceptable to search 20-60S for maxima of
    %both EKE types and ensembles.
    
% 3) DIAGNOSE MAXIMA
[eke_max_lat, eke_max_val, eke_nl_max] = ...
    identify_lats_of_maxima_zeroderiv(lat_hi,eke_0p1,[-60 -20]); %Outputs
    %are 1 x 9 exps x (AQUA, AMIP)
if any(eke_nl_max(:)~=1)
    error('Unexpected number of local maxima!')
end
eke_max_lat = -squeeze(eke_max_lat); %Switch to deg S
eke_max_val =  squeeze(eke_max_val)/1e6; %9 exps x AQ,AM, MJ/m^2