function climate_survey_archive

% 1) CREATE VARIABLES
% a) Data
[data_array, lat] = load_all_data; %This single array holds all data to be
    %plotted, dims 48 lats x (-8:2:8,1x,4x) x AQ,AM x 5 variables
% b) Panel geometry
fwidth   = 19; %Figure width (19 cm is allowed maximum)
fheight  = 23; %Figure height (23 cm is allowed maximum)
bmargin  = .9; %Bottom margin
hmargin  = .3; %Horizontal margin (between rows)
tmargin  = .45; %Top margin
lmargin  = 1.2; %Left margin
vmargin  = .3; %Vertical margin (between columns)
rmargin  = .25;%Right margin
pheight  = 0.2*(fheight-bmargin-tmargin-4*hmargin); %Panel height
pwidth   = 0.5*(fwidth-lmargin-vmargin-rmargin);%Panel width
axpos    = cell(5,2); %Will populate automatically
% c) Other settings
xlims    = [20 60];
ylims    = {[260 300],[-10 15],[0 60],[0 8],[0 1.2]}; %5 variables
fontsize = 8; %Requested by AGU
yaxis_labs = {'925 hPa T (K)','925 hPa U (m/s)','250 hPa U (m/s)',...
    'Eddy MSE transport (PW)','EKE (MJ/m^2)'};
panel_labs = {'(a)','(b)';'(c)','(d)';'(e)','(f)';'(g)','(h)';'(i)','(j)'};
    %One per panel
panel_yscf = [.1 .9 .9 .9 .9];
panel_xval = 22;
cmap       = [parula(9); 0 0 0; 1 0 0]; %-8:2:8,1x,4x

% 2) CHECK IF FIGURE EXISTS
% sname = mfilename;
% sname = sname(16:end); %Get version number
% plot_dir = '/project2/tas1/tmooring/project/tas1/tmooring/plots/echam_diffusivity_plots/';
% if exist([plot_dir 'climate_survey_'  sname '.ps'],'file')
%     error('Planned output plot already exists!')
% end

% 3) DRAW THE FIGURE
% a) Make figure
fh = figure('units','centimeters','position',[3 3 fwidth fheight],...
    'paperunits','centimeters','papersize',[fwidth fheight],...
    'paperposition',[0 0 fwidth fheight],'color','w');
axarr = NaN(5,2); %Axes IDs for current figure
for kk = 1:5 %variables
for jj = 1:2 %AQ,AM
    
% b) Create panel and add metadata lines if necessary
axpos{kk,jj} = [lmargin+((jj-1)*(pwidth+vmargin)) ...
                bmargin+((5-kk)*(pheight+hmargin)) pwidth pheight];
axarr(kk,jj) = axes('units','centimeters','position',axpos{kk,jj});
text(0,0,panel_labs{kk,jj})
pause(1)

% c) Plot data
for ii = 1:11 %exps
plot(lat,data_array(:,ii,jj,kk),'color',cmap(ii,:));
hold on
end %exps

% d) Adjust figure panels and add metadata as needed
set(gca,'fontsize',fontsize,'xlim',xlims,'ylim',ylims{kk},'xtick',...
    20:5:60)
text(panel_xval,ylims{kk}(1)+panel_yscf(kk)*diff(ylims{kk}),...
    panel_labs{kk,jj},'fontsize',fontsize)
if jj==1 %AQ
    ylh = ylabel(yaxis_labs{kk},'fontsize',fontsize);
    set(ylh,'position',[16.5 ylims{kk}(1)+0.5*diff(ylims{kk}) -1])
else %AM
    set(gca,'yticklabel',[])
end
if kk<=4 %all but bottom row
    set(gca,'xticklabel',[])
end
if kk==5
    if jj==1 %AQUA EKE
        set(gca,'xtick',20:5:55)
        xlabel('Latitude (deg)','fontsize',fontsize)
    else %AMIP EKE
        xlabel('Latitude (deg S)','fontsize',fontsize)
    end
end
end %AQ,AM
end %variables

% e) Save figure
legend_todd2([27.5 1.1],5,.5,'-',cmap(1,:),'-8K',fontsize)
legend_todd2([27.5 1.0],5,.5,'-',cmap(2,:),'-6K',fontsize)
legend_todd2([27.5 0.9],5,.5,'-',cmap(3,:),'-4K',fontsize)
legend_todd2([27.5 0.8],5,.5,'-',cmap(4,:),'-2K',fontsize)
legend_todd2([37.5 1.1],5,.5,'-',cmap(5,:),'Prescribed-SST control',fontsize)
legend_todd2([37.5 1.0],5,.5,'-',cmap(6,:),'+2K',fontsize)
legend_todd2([37.5 0.9],5,.5,'-',cmap(7,:),'+4K',fontsize)
legend_todd2([47.5 1.0],5,.5,'-',cmap(8,:),'+6K',fontsize)
legend_todd2([47.5 0.9],5,.5,'-',cmap(9,:),'+8K',fontsize)
legend_todd2([27.5 0.2],5,.5,'-',cmap(10,:),'Slab ocean (359 ppmv CO2)',fontsize)
legend_todd2([27.5 0.1],5,.5,'-',cmap(11,:),'Slab ocean (1436 ppmv CO2)',fontsize)
title(axarr(1,1),'AQUA','fontsize',fontsize,'fontweight','normal');
title(axarr(1,2),'AMIP','fontsize',fontsize,'fontweight','normal');
pause(1)
set(fh,'position',[3 3 fwidth fheight])
pause(1)
for kk = 1:5 %rows (variables)
for jj = 1:2 %cols (AQ,AM)
    set(axarr(kk,jj),'position',axpos{kk,jj})
    pause(1)
end
end
set(gcf,'position',[3 3 fwidth fheight])
pause(1)
print(fh,'-dpsc','climate_survey_archive.ps');
pause(1)
close(fh)

% A) LOAD ALL OF THE DATA
function [data_array, lat] = load_all_data

mfp        = ''; %'/project2/tas1/tmooring/project/tas1/tmooring/mat_files/';
data_array = NaN(96,11,2,5); %96 lats x 11 exps x AQ,AM x 5 variables
lat        = NaN(96,5);      %96 lats x 5 variables
[data_array(:,:,:,1), lat(:,1)] = load_T925(mfp);
[data_array(:,:,:,2), lat(:,2)] = load_U925(mfp);
[data_array(:,:,:,3), lat(:,3)] = load_U250(mfp);
[data_array(:,:,:,4), lat(:,4)] = load_vpmp(mfp);
[data_array(:,:,:,5), lat(:,5)] = load_eke( mfp);
if ~isequal(min(lat,[],2),max(lat,[],2))
    error('Unexpected lats!')
else
    lat = lat(:,1);
end
data_array = data_array(49:96,:,:,:);
lat        = lat(49:96);

% A1) LOAD 925 hPa T
function [T925, lat_T925] = load_T925(mfp)

% a) Create variables
T925_aq    = NaN(96,11);  %96 lats x 11 exps
T925_am    = NaN(96,11,3);%96 lats x 11 exps x J,J,A
lat_T925   = NaN(96,2,2); %96 lats x AQ,AM x fsst,slab
%kh98       = '/project2/tas1/tmooring/NCL_KH98/';
days_jja   = cat(3,30,31,31);
% b) Fixed SST
fsst_aq_ws       = load([mfp 'tiffany_ECHAM20K_925hPa_mean_states_jun26_2018.mat']);
fsst_am_ws       = load([mfp 'tiffany_AMIP16K_925hPa_mean_states_feb23_2018.mat']);
T925_aq(:,1:9)   = squeeze(fsst_aq_ws.mean_states_sym(:,1,2:10));
T925_am(:,1:9,:) = permute(fsst_am_ws.mean_states(:,6:8,1,:),[1 4 2 3]);
lat_T925(:,1,1)  = fsst_aq_ws.lat_ref;
lat_T925(:,2,1)  = fsst_am_ws.lat_ref;
% c) Slab oceans
%T925_aq(:,10)   = mean(mean(ncread([kh98 'KH98_redo_rp000068_eddy_fluxes_means_apr12_2019b.nc'],...
%    't',[1 1 2 1],[inf inf 1 inf]),1),4)';
%T925_aq(:,11)   = mean(mean(ncread([kh98 'KH98_redo_rp000069_means_jun13_2019.nc'],...
%    't',[1 1 2 1],[inf inf 1 inf]),1),4)';
%lat_T925(:,1,2) = ncread([kh98 'KH98_redo_rp000068_eddy_fluxes_means_apr12_2019b.nc'],'lat');
slab_aq_ws         = load([mfp 'reduce_aquaslab_T925_data_volume_nov18_2019.mat']);
T925_aq(:,10:11)   = slab_aq_ws.T925_aqsl;
lat_T925(:,1,2)    = slab_aq_ws.lat_aqsl;
slab_am_ws         = load([mfp 'realslab_925hPa_T_aug10_2019.mat']);
T925_am(:,10:11,:) = flip(permute(slab_am_ws.T(:,6:8,:),[1 3 2]),1);
lat_T925(:,2,2)    = flip(slab_am_ws.lat,1);
% d) Symmetrize AQUA, take JJA means and flip to deg S
T925_aq = 0.5*(T925_aq+flip(T925_aq,1));
T925_am = flip(sum(bsxfun(@times,days_jja,T925_am),3)./sum(days_jja),1);
    %Flip so that inds 49:96 are in SH--T plot confirms this worked for
    %both AMIP and AMIP-SO
% e) Concatenate for return
T925    = cat(3,T925_aq,T925_am);
if ~isequal(min(min(lat_T925,[],2),[],3),max(max(lat_T925,[],2),[],3))
    error('Unexpected T925 lats!')
else
    lat_T925 = lat_T925(:,1,1);
end
    
% A2) LOAD 925 hPa U
function [U925, lat_U925] = load_U925(mfp)

% a) Create variables
U925_aq    = NaN(96,11);  %96 lats x 11 exps
U925_am    = NaN(96,11,3);%96 lats x 11 exps x J,J,A
lat_U925   = NaN(96,2,2); %96 lats x AQ,AM x fsst,slab
days_jja   = cat(3,30,31,31);
% b) Fixed SST
fsst_aq_ws       = load([mfp 'tiffany_ECHAM20K_925hPa_mean_states_jun26_2018.mat']);
fsst_am_ws       = load([mfp 'tiffany_AMIP16K_925hPa_mean_states_feb23_2018.mat']);
U925_aq(:,1:9)   = squeeze(fsst_aq_ws.mean_states_sym(:,5,2:10));
U925_am(:,1:9,:) = permute(fsst_am_ws.mean_states(:,6:8,5,:),[1 4 2 3]);
lat_U925(:,1,1)  = fsst_aq_ws.lat_ref;
lat_U925(:,2,1)  = fsst_am_ws.lat_ref;
% c) Slab oceans
slab_aq_ws         = load([mfp 'aquaslab_925hPa_U_aug09_2019.mat']);
slab_am_ws         = load([mfp 'realslab_925hPa_U_aug08_2019.mat']);
U925_aq(:,10:11)   = flip(slab_aq_ws.U,1); %Match fixed SST
U925_am(:,10:11,:) = flip(permute(slab_am_ws.U(:,6:8,:),[1 3 2]),1);
lat_U925(:,1,2)    = flip(slab_aq_ws.lat,1);
lat_U925(:,2,2)    = flip(slab_am_ws.lat,1);
% d) Symmetrize AQUA, take JJA means and flip to deg S
U925_aq = 0.5*(U925_aq+flip(U925_aq,1));
U925_am = flip(sum(bsxfun(@times,days_jja,U925_am),3)./sum(days_jja),1);
    %Flip so that inds 49:96 are in SH--925 hPa plot confirms this worked
% e) Concatenate for return
U925    = cat(3,U925_aq,U925_am);
if ~isequal(min(min(lat_U925,[],2),[],3),max(max(lat_U925,[],2),[],3))
    error('Unexpected U925 lats!')
else
    lat_U925 = lat_U925(:,1,1);
end

% A3) LOAD 250 hPa U
function [U250, lat_U250] = load_U250(mfp)

% a) Create variables
U250_aq    = NaN(96,11);  %96 lats x 11 exps
U250_am    = NaN(96,11,3);%96 lats x 11 exps x J,J,A
lat_U250   = NaN(96,2,2); %96 lats x AQ,AM x fsst,slab
days_jja   = cat(3,30,31,31);
% b) Fixed SST
fsst_aq_ws       = load([mfp 'ECHAM16K_250hPa_U_aug09_2019.mat']);
fsst_am_ws       = load([mfp 'AMIP16K_250hPa_U_aug09_2019.mat']);
U250_aq(:,1:9)   = fsst_aq_ws.U;
U250_am(:,1:9,:) = permute(fsst_am_ws.U(:,6:8,:),[1 3 2]);
lat_U250(:,1,1)  = fsst_aq_ws.lat_ref;
lat_U250(:,2,1)  = fsst_am_ws.lat_ref;
% c) Slab oceans
slab_aq_ws         = load([mfp 'aquaslab_250hPa_U_aug09_2019.mat']);
slab_am_ws         = load([mfp 'realslab_250hPa_U_aug08_2019.mat']);
U250_aq(:,10:11)   = flip(slab_aq_ws.U,1); %Match fixed SST
U250_am(:,10:11,:) = flip(permute(slab_am_ws.U(:,6:8,:),[1 3 2]),1);
lat_U250(:,1,2)    = flip(slab_aq_ws.lat,1);
lat_U250(:,2,2)    = flip(slab_am_ws.lat,1);
% d) Symmetrize AQUA, take JJA means and flip to deg S
U250_aq = 0.5*(U250_aq+flip(U250_aq,1));
U250_am = flip(sum(bsxfun(@times,days_jja,U250_am),3)./sum(days_jja),1);
    %Flip so that inds 49:96 are in SH--250 hPa plot confirms this worked
% e) Concatenate for return
U250    = cat(3,U250_aq,U250_am);
if ~isequal(min(min(lat_U250,[],2),[],3),max(max(lat_U250,[],2),[],3))
    error('Unexpected U250 lats!')
else
    lat_U250 = lat_U250(:,1,1);
end

% A4) LOAD EDDY MSE TRANSPORT
function [vpmp, lat_vpmp] = load_vpmp(mfp)

% a) Create variables
vpmp_aq  = NaN(96,11);  %96 lats x 11 exps
vpmp_am  = NaN(96,11,3);%96 lats x 11 exps x J,J,A
lat_vpmp = NaN(96,2,2); %96 lats x AQ,AM x fsst,slab
days_jja = cat(3,30,31,31);
% b) Fixed SST
fsst_aq_ws         = load([mfp 'tiffany_ECHAM16K_tm_trans_attrib_jul09_2018.mat']);
fsst_am_ws         = load([mfp 'tiffany_AMIP16K_tm_trans_attrib_jul09_2018.mat']);
vpmp_aq(:,1:9)     = flip(squeeze(fsst_aq_ws.tbasic_1p9(:,4,:)),1);
vpmp_am(:,1:9,:)   = flip(permute(fsst_am_ws.tbasic_1p9(:,6:8,4,:),[1 4 2 3]),1);
lat_vpmp(:,1,1)    = flip(fsst_aq_ws.lat_ref,1);
lat_vpmp(:,2,1)    = flip(fsst_am_ws.lat_ref,1);
% c) Slab oceans
slab_aq_ws         = load([mfp 'tiffany_aquaslab_1x4xCO2_tm_trans_attrib_jun13_2019.mat']);
slab_am_ws         = load([mfp 'tiffany_realslab_1x4xCO2_tm_trans_attrib_JJA_jun13_2019.mat']);
vpmp_aq(:,10:11)   = squeeze(mean(slab_aq_ws.tfilt_1p9(:,:,4,:),2));
vpmp_am(:,10:11,:) = permute(slab_am_ws.tfilt_1p9(:,:,4,:),[1 4 2 3]);
lat_vpmp(:,1,2)    = slab_aq_ws.lat;
lat_vpmp(:,2,2)    = slab_am_ws.lat;
% d) Symmetrize AQUA, take JJA means and flip to deg S
vpmp_aq = 0.5*(vpmp_aq-flip(vpmp_aq,1));
vpmp_am = -flip(sum(bsxfun(@times,days_jja,vpmp_am),3)./sum(days_jja),1);
    %Also change sign so +ve is poleward in SH--plotting confirms that the
    %sign change and latitude flip have worked as expected.
% e) Concatenate for return
vpmp = cat(3,vpmp_aq,vpmp_am)/1e15;
if ~isequal(min(min(lat_vpmp,[],2),[],3),max(max(lat_vpmp,[],2),[],3))
    error('Unexpected vpmp lats!')
else
    lat_vpmp = lat_vpmp(:,1,1);
end

% A5) LOAD EDDY KINETIC ENERGY
function [eke, lat_eke] = load_eke(mfp)

% a) Create variables
eke_aq   = NaN(96,11);  %96 lats x 11 exps
eke_am   = NaN(96,11,3);%96 lats x 11 exps x J,J,A
lat_eke  = NaN(96,2,2); %96 lats x AQ,AM x fsst,slab
days_jja = cat(3,30,31,31);
% b) Fixed SST
fsst_aq_ws        = load([mfp 'ECHAM20K_col_int_eke_comps_jun27_2018.mat']);
fsst_am_ws        = load([mfp 'AMIP16K_col_int_eke_comps_jun29_2018.mat']);
eke_aq(:,1:9)     = squeeze(0.5*sum(mean(mean(fsst_aq_ws.xp2(:,:,:,:,2:10),2),3),4));
eke_am(:,1:9,:)   = permute(0.5*sum(mean(fsst_am_ws.xp2(:,6:8,:,:,:),3),4),[1 5 2:4]);
lat_eke(:,1,1)    = fsst_aq_ws.lat;
lat_eke(:,2,1)    = fsst_am_ws.lat;
% c) Slab oceans
%slab_aq_ws        = load([mfp 'aquaslab_1x4xCO2_lev_res_eke_comps_aug08_2019.mat']);
%slab_am_ws        = load([mfp 'realslab_1x4xCO2_lev_res_eke_comps_aug08_2019.mat']);
%eke_aq(:,10:11)   = squeeze(0.5*sum(mean(mean(sum(slab_aq_ws.xp2,2),3),4),5));
%eke_am(:,10:11,:) = permute(0.5*sum(mean(sum(slab_am_ws.xp2(:,:,6:8,:,:,:),2),4),5),[1 6 3 2 4 5]);
%lat_eke(:,1,2)    = slab_aq_ws.lat;
%lat_eke(:,2,2)    = slab_am_ws.lat;
slab_ws            = load([mfp 'reduce_slab_EKE_data_volume_nov18_2019.mat']);
eke_aq(:,10:11)    = squeeze(0.5*sum(mean(mean(slab_ws.eke_aqsl_cs,3),4),5));
eke_am(:,10:11,:)  = permute(0.5*sum(mean(slab_ws.eke_amsl_cs(:,:,6:8,:,:,:),4),5),[1 6 3 2 4 5]);
lat_eke(:,1,2)     = slab_ws.slab_aq_meta.lat;
lat_eke(:,2,2)     = slab_ws.slab_am_meta.lat;
% d) Symmetrize AQUA, take JJA means and flip to deg S
eke_aq = 0.5*(eke_aq+flip(eke_aq,1));
eke_am = flip(sum(bsxfun(@times,days_jja,eke_am),3)./sum(days_jja),1);
    %Also change sign so +ve is poleward in SH--EKE panels of plot confirm
    %that the sign change and latitude flip have worked as expected.
% e) Concatenate for return
eke     = flip(cat(3,eke_aq,eke_am),1)/1e6; %MJ/m^2
lat_eke = flip(lat_eke,1); %Match other lats
if ~isequal(min(min(lat_eke,[],2),[],3),max(max(lat_eke,[],2),[],3))
    error('Unexpected eke lats!')
else
    lat_eke = lat_eke(:,1,1);
end