function [max_lat, max_val, n_local_max] = ...
    identify_lats_of_maxima_zeroderiv(lat,fld,latlim)

% This function computes latitudes of the maxima of some meridionally-
% varying input field.  This is done by estimating latitudes where the
% meridional derivative of the field goes to zero.  If no maximum can be
% found for a given input meridional profile, this function instead returns
% the latitude at which an estimate of the slope of the meridional profile
% has its smallest magnitude.  This could be a local minimum, a saddle
% point, or the closest thing to a saddle point available in the profile.
%
% INPUT:
% lat          Column vector of latitudes in degrees N
% fld          Array containing the field whose maxima we are seeking.  Can
%                  have any number of dimensions, but the first must be
%                  latitude
% latlim       Two-element row vector of latitude range in which to search
%                  for the local maximum
%
% OUTPUT:
% max_lat      Array containing the latitudes of maxima (or minima of slope
%                  magnitude, if there is no maximum).  First dimension
%                  has length 1, other dimensions are the same as in fld
% max_val      Array containing values of the field associated with the
%                  maxima, same dimensionality as max_lat.  See code for
%                  exactly how these are computed
% n_local_max  The number of local maxima in the region of interest, same
%                  dimensions as max_lat.  This will be 0 if we are forced
%                  to report a latitude of smallest magnitude slope

% 1) VALIDATE INPUT
dlat = diff(lat,1,1);
if all(sign(dlat)==(-1))
    lat = flip(lat,1); %Rearrange so lats increase w/ increasing index
    fld = flip(fld,1);
elseif ~all(sign(dlat)==1)
    error('lat is not a valid latitude vector!')
end

% 2) RESHAPE INPUT FLD
fld_dims = size(fld);
fld      = reshape(fld,[fld_dims(1) prod(fld_dims(2:end))]); %Reshape so
    %that the entirety of fld fits onto one page
    
% 3) CREATE OUTPUT VARIABLES
max_lat     = NaN(1,prod(fld_dims(2:end))); %Will rearrange to output
max_val     = max_lat; %dimensionality later
n_local_max = max_lat;

% 4) DO THE MAXIMUM SEARCH
for ii = 1:size(fld,2)
    %iterate over all meridional profiles
    fldc    = fld(:,ii); %Select current meridional profile
    nxt_is_smaller = [   fldc(1:(end-1))>fldc(2:end); 0]; %For each
    pvs_is_smaller = [0; fldc(1:(end-1))<fldc(2:end)   ]; %latitude point,
        %identify whether the previous and next points are smaller
        %(basically the definition of a local maximum).  The first (last)
        %point doesn't have a previous (next) point, so it can't be a local
        %maximum.
    is_local_max = find(nxt_is_smaller & pvs_is_smaller); %ID local maxima
    local_max_lats = NaN(size(is_local_max)); %Latitudes of local maxima
    local_max_vals = local_max_lats; %Values at local maxima on raw grid
    for jj = 1:numel(is_local_max)
        %iterate over all local maxima
        lat_sub  = lat((is_local_max(jj)-1):(is_local_max(jj)+1));
        fldc_sub = fldc((is_local_max(jj)-1):(is_local_max(jj)+1));
        local_max_vals(jj) = fldc(is_local_max(jj));
        slps     = diff(fldc_sub,1,1)./diff(lat_sub,1,1); %Slopes of the
            %profile adjacent to the maximum in the raw data
        laths    = 0.5*(lat_sub(1:2)+lat_sub(2:3)); %The half latitudes at
            %which the slopes are defined
        local_max_lats(jj) = ...
            laths(1)-slps(1)*(diff(laths,1,1)/diff(slps,1,1)); %Using
            %linear interpolation, find the latitude where the meridional
            %slope of the profile is zero and store this information
    end
    local_max_lats_ir = local_max_lats((local_max_lats>=latlim(1)) & ...
                                       (local_max_lats<=latlim(2)));
    local_max_vals_ir = local_max_vals((local_max_lats>=latlim(1)) & ...
                                       (local_max_lats<=latlim(2)));
        %Retain only those latitudes (and values) of local maxima actually
        %located within the region of interest
    if numel(local_max_lats_ir)==1
        %There is one local maximum in the region of interest
        max_lat(1,ii)     = local_max_lats_ir;
        max_val(1,ii)     = local_max_vals_ir;
        n_local_max(1,ii) = 1;
    elseif numel(local_max_lats_ir)>1
        %There is more than one local maximum in the region of interest
        max_lat(1,ii)     = ...
            local_max_lats_ir(local_max_vals_ir==max(local_max_vals_ir));
        max_val(1,ii)     = max(local_max_vals_ir);
        n_local_max(1,ii) = numel(local_max_lats_ir);
    else
        %There is no actual local maximum in the region of interest.  We
        %need to find the latitude with the smallest-magnitude slope and
        %then select that.
        slp_all  = diff(fldc,1,1)./diff(lat,1,1); %Slopes on half latitudes
        lath_all = 0.5*(lat(1:(end-1))+lat(2:end)); %Half latitudes
        fldh     = interp1(lat,fldc,lath_all); %Linearly interpolate data
            %to the half latitudes
        slp_all  =  slp_all((lath_all>=latlim(1)) & (lath_all<=latlim(2)));
        fldh     =     fldh((lath_all>=latlim(1)) & (lath_all<=latlim(2)));
        lath_all = lath_all((lath_all>=latlim(1)) & (lath_all<=latlim(2)));
            %Subset to the latitude range of interest
        abs_slp_all       = abs(slp_all); %Slope magnitudes
        max_lat(1,ii)     = lath_all(abs_slp_all==min(abs_slp_all));
        max_val(1,ii)     = fldh(abs_slp_all==min(abs_slp_all));
        n_local_max(1,ii) = 0;
    end
end

% 5) RESHAPE VARIABLES FOR RETURN
max_lat     = reshape(max_lat,[1 fld_dims(2:end)]);
max_val     = reshape(max_val,[1 fld_dims(2:end)]);
n_local_max = reshape(n_local_max,[1 fld_dims(2:end)]);